/// @file eventHandler.h
///
/// @brief Describes the @ref EventHandler class
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see Udev

#ifndef __EVENT_HANDLER__
#define __EVENT_HANDLER__

#include <thread>

namespace adit {
namespace uspi {

/// @brief Event handler class singleton
///
/// The event handler initializes an epoll file descriptor, creates a dedicated
/// thread for polling and allows registration for new event.
/// The registration currently only accepts a C-style fd + callback parameters.
/// This could be extended to have a more object oriented version (event class)
class EventHandler {
    /// Epoll file descriptor
    int mEfd;
    /// Shall we stop the loop ?
    int mTerminate;
    /// The event thread
    std::thread mEventThread;

    EventHandler();
    ~EventHandler();
    /// Wait for event on one of the registered file handler.
    int eventWait(void);

    /// The waiter for the thread
    void wait(void);
public:
    /// The instance provider
    static EventHandler& getInstance()
    {
        static EventHandler eh;
        return eh;
    };

    /// The registration method.
    int registerEvent(int fd, void *cb);
    /// The active loop. Shall be executed if no thread is expected to be used.
    void eventLoop(void);
    /// The thread creator
    void startEventLoop(void);
    /// The thread terminator
    void terminateEventLoop(bool wait);
};

} // namespace uspi
} // namespace adit
#endif
